const API_BASE = "../api";

const state = {
  cart: [],
  menu: [],
  categories: [],
  user: null,
};

const PLACEHOLDER_IMG = "data:image/svg+xml;base64,PHN2ZyB4bWxucz0naHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmcnIHdpZHRoPSczMDAnIGhlaWdodD0nMzAwJz48cmVjdCB3aWR0aD0nMTAwJScgaGVpZ2h0PScxMDAlJyBmaWxsPScjZTJlOGYwJy8+PHRleHQgeD0nNTAlJyB5PSc1MCUnIGR5PScuMzVlbScgdGV4dC1hbmNob3I9J21pZGRsZScgZmlsbD0nIzZiNzI4MCcgZm9udC1zaXplPScyMCc+Tm8gSW1hZ2U8L3RleHQ+PC9zdmc+";

function qs(selector, parent = document) { return parent.querySelector(selector); }
function qsa(selector, parent = document) { return [...parent.querySelectorAll(selector)]; }
function esc(value = "") {
  return String(value)
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#39;");
}

async function api(path, options = {}) {
  const config = {
    headers: { "Content-Type": "application/json" },
    credentials: "same-origin",
    ...options,
  };

  if (config.body && typeof config.body !== "string") {
    config.body = JSON.stringify(config.body);
  }

  const res = await fetch(`${API_BASE}${path}`, config);
  if (res.headers.get("content-type")?.includes("application/json")) {
    const payload = await res.json();
    if (payload.status === "error") throw new Error(payload.message || "Request failed");
    return payload.data;
  }
  throw new Error("Invalid server response format");
}

function toast(message, isError = false) {
  const el = qs("#toast");
  if (!el) return;
  el.textContent = message;
  el.style.background = isError ? "#991b1b" : "#1d4ed8";
  el.classList.remove("hidden");
  setTimeout(() => el.classList.add("hidden"), 2500);
}

function downloadBase64File(fileName, mimeType, base64Content) {
  const byteString = atob(base64Content);
  const len = byteString.length;
  const bytes = new Uint8Array(len);
  for (let i = 0; i < len; i += 1) {
    bytes[i] = byteString.charCodeAt(i);
  }
  const blob = new Blob([bytes], { type: mimeType });
  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = fileName;
  document.body.appendChild(a);
  a.click();
  a.remove();
  URL.revokeObjectURL(url);
}

function getQueryParam(key) {
  const p = new URLSearchParams(window.location.search);
  return p.get(key);
}

function attachCommonActions() {
  const logoutBtn = qs("#logoutBtn");
  if (logoutBtn) {
    logoutBtn.addEventListener("click", async () => {
      try {
        await api("/auth/logout.php", { method: "POST", body: {} });
        window.location.href = "index.html";
      } catch (e) {
        toast(e.message, true);
      }
    });
  }
}

async function initIndexPage() {
  const form = qs("#loginForm");
  if (!form) return;

  form.addEventListener("submit", async (e) => {
    e.preventDefault();
    const username = qs("#username").value.trim();
    const password = qs("#password").value;

    try {
      const data = await api("/auth/login.php", { method: "POST", body: { username, password } });
      const role = data.user.role;
      if (role === "admin") window.location.href = "admin.html";
      else if (role === "kitchen") window.location.href = "kitchen.html";
      else window.location.href = "waiter.html";
    } catch (err) {
      toast(err.message, true);
    }
  });
}

function renderCart() {
  const cartWrap = qs("#cartList");
  if (!cartWrap) return;

  if (!state.cart.length) {
    cartWrap.innerHTML = `<p class="muted">Cart is empty.</p>`;
    qs("#cartTotal").textContent = "0.00";
    return;
  }

  let total = 0;
  cartWrap.innerHTML = state.cart.map((item, idx) => {
    total += item.price * item.quantity;
    return `<div class="row" style="justify-content:space-between;border-bottom:1px solid #dbe3ef;padding:8px 0;">
      <div>
        <div><strong>${esc(item.name)}</strong></div>
        <small>₹${item.price.toFixed(2)} x ${item.quantity}</small>
      </div>
      <button data-remove="${idx}" class="btn-danger">Remove</button>
    </div>`;
  }).join("");

  qs("#cartTotal").textContent = total.toFixed(2);

  qsa("[data-remove]", cartWrap).forEach(btn => {
    btn.addEventListener("click", () => {
      const idx = Number(btn.getAttribute("data-remove"));
      state.cart.splice(idx, 1);
      renderCart();
    });
  });
}

function addToCart(itemId) {
  const item = state.menu.find(i => i.id === itemId);
  if (!item || !item.is_currently_available) return;
  const existing = state.cart.find(i => i.menu_item_id === item.id);
  if (existing) existing.quantity += 1;
  else state.cart.push({ menu_item_id: item.id, name: item.name, price: item.price, quantity: 1 });
  renderCart();
  toast("Added to cart");
}

function renderMenu() {
  const host = qs("#menuCategories");
  if (!host) return;

  host.innerHTML = state.categories.map(cat => {
    const items = cat.items || [];
    const cards = items.map(item => `
      <div class="menu-item">
        <img src="${esc(item.image_path || PLACEHOLDER_IMG)}" alt="${esc(item.name)}" onerror="this.src='${PLACEHOLDER_IMG}'" />
        <div>
          <div class="row" style="justify-content:space-between;align-items:flex-start;">
            <h4>${esc(item.name)}</h4>
            <strong>₹${Number(item.price).toFixed(2)}</strong>
          </div>
          <p class="muted">${esc(item.description || '')}</p>
          <p class="muted">Tags: ${esc(item.tags || '-')} | Dietary: ${esc(item.dietary_tags || '-')}</p>
          <div class="row" style="justify-content:space-between;">
            <span class="badge status-${esc(item.availability_state)}">${esc(item.availability_state.replaceAll('_',' '))}</span>
            <button ${item.is_currently_available ? '' : 'disabled'} data-add="${item.id}" class="btn">Add</button>
          </div>
        </div>
      </div>`).join("");

    return `<section class="card"><h3>${esc(cat.name)}</h3><div class="grid">${cards || '<p class="muted">No items</p>'}</div></section>`;
  }).join("");

  qsa("[data-add]").forEach(btn => btn.addEventListener("click", () => addToCart(Number(btn.getAttribute("data-add")))));
}

async function initMenuPage() {
  const tableFromQuery = getQueryParam("table") || "";
  const tableInput = qs("#tableNumber");
  if (tableInput && tableFromQuery) tableInput.value = tableFromQuery;

  try {
    const data = await api("/menu/public.php");
    state.categories = data.categories;
    state.menu = data.categories.flatMap(c => c.items || []);
    renderMenu();
    renderCart();
  } catch (e) {
    toast(e.message, true);
  }

  qs("#placeOrderBtn")?.addEventListener("click", async () => {
    const tableNumber = qs("#tableNumber").value.trim();
    const special_notes = qs("#orderNotes").value.trim();

    if (!tableNumber || state.cart.length === 0) {
      toast("Enter table number and add items", true);
      return;
    }

    try {
      const data = await api("/orders/place.php", {
        method: "POST",
        body: {
          table_number: tableNumber,
          special_notes,
          items: state.cart.map(i => ({ menu_item_id: i.menu_item_id, quantity: i.quantity }))
        }
      });
      qs("#orderStatus").innerHTML = `<span class='badge status-received'>Order #${data.order_id} confirmed</span>`;
      state.cart = [];
      renderCart();
    } catch (e) {
      toast(e.message, true);
    }
  });

  qs("#callWaiterBtn")?.addEventListener("click", async () => {
    const table_number = qs("#tableNumber").value.trim();
    if (!table_number) return toast("Enter table number", true);
    try {
      await api("/orders/call_waiter.php", { method: "POST", body: { table_number } });
      toast("Waiter has been notified");
    } catch (e) {
      toast(e.message, true);
    }
  });

  const refreshOrder = async () => {
    const tableNumber = qs("#tableNumber").value.trim();
    if (!tableNumber) return;
    try {
      const data = await api(`/orders/by_table.php?table_number=${encodeURIComponent(tableNumber)}`);
      if (!data.order) return;
      const order = data.order;
      qs("#activeOrderItems").innerHTML = order.items.map(i => `<li>${esc(i.name)} x ${i.quantity} <span class='badge status-${esc(i.status)}'>${esc(i.status)}</span></li>`).join("");
      qs("#activeOrderMeta").textContent = `Order #${order.order_id} | ${order.order_status}`;
    } catch {
      // silent polling fail
    }
  };

  await refreshOrder();
  setInterval(refreshOrder, 3000);
}

function urgencyClass(minutes) {
  if (minutes >= 20) return "urgent";
  if (minutes >= 10) return "warning";
  return "normal";
}

async function loadKitchenOrders() {
  try {
    const data = await api("/orders/active.php");
    const host = qs("#kitchenOrders");
    if (!host) return;

    host.innerHTML = data.orders.map(order => `
      <article class="card kitchen-order ${urgencyClass(order.minutes_elapsed)}">
        <div class="row" style="justify-content:space-between;">
          <h3>Table ${esc(order.table_number)} • Order #${order.order_id}</h3>
          <span class="badge status-${esc(order.order_status)}">${esc(order.order_status)}</span>
        </div>
        <p class="muted">Elapsed: ${order.minutes_elapsed} min ${order.order_notes ? `| Note: ${esc(order.order_notes)}` : ""}</p>
        <div class="grid">
          ${order.items.map(item => `<div class='row' style='justify-content:space-between;border-top:1px dashed #dbe3ef;padding-top:8px;'>
              <div>${esc(item.name)} x ${item.quantity} ${item.notes ? `<small>(${esc(item.notes)})</small>` : ""}</div>
              <div class='row'>
                <span class='badge status-${esc(item.status)}'>${esc(item.status)}</span>
                <select data-item-status='${item.order_item_id}' data-order='${order.order_id}'>
                  <option value='received' ${item.status==='received'?'selected':''}>Received</option>
                  <option value='preparing' ${item.status==='preparing'?'selected':''}>Preparing</option>
                  <option value='ready' ${item.status==='ready'?'selected':''}>Ready</option>
                  <option value='served' ${item.status==='served'?'selected':''}>Served</option>
                </select>
              </div>
            </div>`).join("")}
        </div>
      </article>
    `).join("") || `<p class='muted'>No active orders.</p>`;

    qsa("[data-item-status]", host).forEach(select => {
      select.addEventListener("change", async () => {
        try {
          await api("/orders/update_status.php", {
            method: "POST",
            body: {
              order_id: Number(select.getAttribute("data-order")),
              order_item_id: Number(select.getAttribute("data-item-status")),
              status: select.value,
            },
          });
          toast("Status updated");
          loadKitchenOrders();
        } catch (e) {
          toast(e.message, true);
        }
      });
    });
  } catch (e) {
    toast(e.message, true);
  }
}

async function ensureRole(roles) {
  try {
    const me = await api("/auth/me.php");
    state.user = me.user;
    if (!roles.includes(me.user.role)) {
      window.location.href = "index.html";
      return false;
    }
    qs("#whoami") && (qs("#whoami").textContent = `${me.user.full_name} (${me.user.role})`);
    return true;
  } catch {
    window.location.href = "index.html";
    return false;
  }
}

async function initKitchenPage() {
  if (!(await ensureRole(["kitchen", "admin"]))) return;
  await loadKitchenOrders();
  setInterval(loadKitchenOrders, 3000);
}

async function loadTables(selector = "#tablesGrid") {
  const host = qs(selector);
  if (!host) return;
  try {
    const data = await api("/tables/list.php");
    host.innerHTML = data.tables.map(t => `
      <div class='card'>
        <div class='row' style='justify-content:space-between;'>
          <h4>Table ${esc(t.table_number)}</h4>
          <span class='badge status-${esc(t.status)}'>${esc(t.status.replaceAll('_',' '))}</span>
        </div>
        <p class='muted'>Seats: ${t.seats} | Waiter: ${esc(t.assigned_waiter_name || '-')}</p>
        <div class='row'>
          <button class='btn-outline' data-clean='${t.id}'>Needs cleaning</button>
          <button class='btn-secondary' data-close='${t.id}'>Close</button>
        </div>
      </div>`).join("");

    qsa("[data-clean]", host).forEach(btn => btn.addEventListener("click", () => tableAction("mark_cleaning", { table_id: Number(btn.dataset.clean) })));
    qsa("[data-close]", host).forEach(btn => btn.addEventListener("click", () => tableAction("close_table", { table_id: Number(btn.dataset.close) })));
  } catch (e) {
    toast(e.message, true);
  }
}

async function tableAction(action, extra = {}) {
  try {
    await api("/tables/manage.php", { method: "POST", body: { action, ...extra } });
    toast("Table updated");
    loadTables("#tablesGrid");
    loadTables("#waiterTablesGrid");
  } catch (e) {
    toast(e.message, true);
  }
}

async function loadSummary() {
  try {
    const data = await api("/billing/summary.php");
    qs("#revToday") && (qs("#revToday").textContent = `₹${Number(data.revenue.today).toFixed(2)}`);
    qs("#revWeek") && (qs("#revWeek").textContent = `₹${Number(data.revenue.week).toFixed(2)}`);
    qs("#revMonth") && (qs("#revMonth").textContent = `₹${Number(data.revenue.month).toFixed(2)}`);

    const top = qs("#topItems");
    if (top) {
      top.innerHTML = data.top_items.map(i => `<li>${esc(i.item_name)} (${i.total_qty})</li>`).join("") || "<li>No data</li>";
    }
  } catch (e) {
    toast(e.message, true);
  }
}

async function loadHistory() {
  try {
    const from = qs("#filterFrom")?.value || "";
    const to = qs("#filterTo")?.value || "";
    const status = qs("#filterStatus")?.value || "";
    const data = await api(`/orders/history.php?from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}&status=${encodeURIComponent(status)}`);

    const host = qs("#historyBody");
    if (!host) return;
    host.innerHTML = data.orders.map(o => `<tr>
      <td>${o.id}</td>
      <td>${esc(o.table_number)}</td>
      <td><span class='badge status-${esc(o.status)}'>${esc(o.status)}</span></td>
      <td>₹${Number(o.grand_total).toFixed(2)}</td>
      <td>${o.is_paid ? "Yes" : "No"}</td>
      <td>${esc(o.created_at)}</td>
      <td><button class='btn-outline' data-bill='${o.id}'>Open Bill</button></td>
    </tr>`).join("");

    qsa("[data-bill]", host).forEach(btn => btn.addEventListener("click", () => openBill(Number(btn.dataset.bill))));
  } catch (e) {
    toast(e.message, true);
  }
}

async function openBill(orderId) {
  try {
    const data = await api(`/billing/get.php?order_id=${orderId}`);
    qs("#billOrderId").textContent = String(orderId);

    qs("#billItems").innerHTML = data.items.map(item => `<li>${esc(item.item_name_snapshot)} x ${item.quantity} = ₹${Number(item.line_total).toFixed(2)}</li>`).join("");
    qs("#manualCharges").innerHTML = data.manual_charges.map(c => `<li>${esc(c.title)}: ₹${Number(c.amount).toFixed(2)}</li>`).join("") || "<li>None</li>";

    qs("#subtotal").textContent = Number(data.totals.subtotal).toFixed(2);
    qs("#tax").textContent = Number(data.totals.tax_amount).toFixed(2);
    qs("#service").textContent = Number(data.totals.service_charge_amount).toFixed(2);
    qs("#discount").textContent = Number(data.totals.discount_amount).toFixed(2);
    qs("#grand").textContent = Number(data.totals.grand_total).toFixed(2);

    qs("#billingPanel").classList.remove("hidden");
    qs("#billOrderInput").value = orderId;
  } catch (e) {
    toast(e.message, true);
  }
}

async function loadMenuCrud() {
  try {
    const [cats, items] = await Promise.all([api("/menu/categories.php"), api("/menu/items.php")]);
    const catHost = qs("#adminCategories");
    const itemHost = qs("#adminItems");
    const catSelect = qs("#itemCategory");

    if (catHost) {
      catHost.innerHTML = cats.categories.map(c => `<div class='row' style='justify-content:space-between;border-bottom:1px solid #dbe3ef;padding:6px 0;'>
        <strong>${esc(c.name)}</strong>
        <button class='btn-danger' data-del-cat='${c.id}'>Delete</button>
      </div>`).join("") || "<p class='muted'>No categories.</p>";
      qsa("[data-del-cat]", catHost).forEach(btn => btn.addEventListener("click", async () => {
        await api("/menu/categories.php", { method: "POST", body: { action: "delete", id: Number(btn.dataset.delCat) } });
        loadMenuCrud();
      }));
    }

    if (catSelect) {
      catSelect.innerHTML = cats.categories.map(c => `<option value='${c.id}'>${esc(c.name)}</option>`).join("");
    }

    if (itemHost) {
      itemHost.innerHTML = items.items.map(i => `<div class='row' style='justify-content:space-between;border-bottom:1px solid #dbe3ef;padding:6px 0;'>
        <span>${esc(i.name)} (₹${Number(i.price).toFixed(2)})</span>
        <button class='btn-danger' data-del-item='${i.id}'>Delete</button>
      </div>`).join("") || "<p class='muted'>No items.</p>";
      qsa("[data-del-item]", itemHost).forEach(btn => btn.addEventListener("click", async () => {
        await api("/menu/items.php", { method: "POST", body: { action: "delete", id: Number(btn.dataset.delItem) } });
        loadMenuCrud();
      }));
    }
  } catch (e) {
    toast(e.message, true);
  }
}

async function loadQrList() {
  try {
    const [tablesData, qrData] = await Promise.all([api("/tables/list.php"), api("/qr/list.php")]);
    const select = qs("#qrTableSelect");
    if (select) select.innerHTML = tablesData.tables.map(t => `<option value='${t.id}'>Table ${esc(t.table_number)}</option>`).join("");

    const host = qs("#qrList");
    if (host) {
      host.innerHTML = qrData.qr_codes.map(q => `<div class='card'>
        <div class='row' style='justify-content:space-between;'>
          <strong>Table ${esc(q.table_number)}</strong>
          <a class='btn-outline' href='${esc(q.image_path)}' download>Download PNG</a>
        </div>
        <img src='${esc(q.image_path)}' alt='QR' style='width:160px;height:160px;object-fit:contain;margin-top:8px;' />
      </div>`).join("") || "<p class='muted'>No QR generated yet.</p>";
    }
  } catch (e) {
    toast(e.message, true);
  }
}

async function initAdminPage() {
  if (!(await ensureRole(["admin"]))) return;
  await Promise.all([loadSummary(), loadKitchenOrders(), loadTables(), loadHistory(), loadMenuCrud(), loadQrList()]);

  qs("#historyFilterBtn")?.addEventListener("click", loadHistory);

  qs("#addCategoryForm")?.addEventListener("submit", async e => {
    e.preventDefault();
    await api("/menu/categories.php", { method: "POST", body: { action: "create", name: qs("#categoryName").value.trim(), display_order: 0 } });
    qs("#categoryName").value = "";
    loadMenuCrud();
  });

  qs("#addItemForm")?.addEventListener("submit", async e => {
    e.preventDefault();
    const fd = new FormData();
    fd.append("action", "create");
    fd.append("category_id", qs("#itemCategory").value);
    fd.append("name", qs("#itemName").value.trim());
    fd.append("description", qs("#itemDescription").value.trim());
    fd.append("price", qs("#itemPrice").value);
    fd.append("availability_state", qs("#itemAvailability").value);
    const file = qs("#itemImage").files[0];
    if (file) fd.append("image", file);

    const res = await fetch(`${API_BASE}/menu/items.php`, { method: "POST", credentials: "same-origin", body: fd });
    const payload = await res.json();
    if (payload.status === "error") throw new Error(payload.message);
    qs("#addItemForm").reset();
    loadMenuCrud();
  });

  qs("#addChargeBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#billOrderInput").value);
    const title = qs("#chargeTitle").value.trim();
    const amount = Number(qs("#chargeAmount").value);
    await api("/billing/manage.php", { method: "POST", body: { action: "manual_charge", order_id, title, amount } });
    openBill(order_id);
  });

  qs("#discountBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#billOrderInput").value);
    const discount_amount = Number(qs("#discountInput").value || 0);
    await api("/billing/manage.php", { method: "POST", body: { action: "set_discount", order_id, discount_amount } });
    openBill(order_id);
  });

  qs("#payBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#billOrderInput").value);
    const amount = Number(qs("#payAmount").value);
    const payment_method = qs("#payMethod").value;
    await api("/billing/manage.php", { method: "POST", body: { action: "mark_paid", order_id, amount, payment_method } });
    toast("Payment recorded");
    loadHistory();
    loadTables();
  });

  qs("#splitBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#billOrderInput").value);
    const split_count = Number(qs("#splitCount").value || 1);
    await api("/billing/manage.php", { method: "POST", body: { action: "set_split", order_id, split_count } });
    openBill(order_id);
  });

  qs("#settingsBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#billOrderInput").value);
    const tax_percent = Number(qs("#taxPercent").value || 0);
    const service_charge_percent = Number(qs("#servicePercent").value || 0);
    await api("/billing/manage.php", { method: "POST", body: { action: "set_settings", order_id, tax_percent, service_charge_percent } });
    openBill(order_id);
  });

  qs("#receiptBtn")?.addEventListener("click", () => {
    const orderId = Number(qs("#billOrderInput").value);
    if (orderId > 0) {
      window.open(`receipt.html?order_id=${orderId}`, "_blank");
    }
  });

  qs("#exportCsvBtn")?.addEventListener("click", () => {
    (async () => {
      try {
        const from = qs("#filterFrom").value || "";
        const to = qs("#filterTo").value || "";
        const data = await api(`/billing/export_csv.php?from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}`);
        downloadBase64File(data.file_name, data.mime_type, data.base64_content);
      } catch (e) {
        toast(e.message, true);
      }
    })();
  });

  qs("#exportPdfBtn")?.addEventListener("click", () => {
    (async () => {
      try {
        const from = qs("#filterFrom").value || "";
        const to = qs("#filterTo").value || "";
        const data = await api(`/billing/export_pdf.php?from=${encodeURIComponent(from)}&to=${encodeURIComponent(to)}`);
        downloadBase64File(data.file_name, data.mime_type, data.base64_content);
      } catch (e) {
        toast(e.message, true);
      }
    })();
  });

  qs("#generateQrBtn")?.addEventListener("click", async () => {
    const table_id = Number(qs("#qrTableSelect").value);
    const domain = qs("#qrDomain").value.trim();
    await api("/qr/generate.php", { method: "POST", body: { table_id, domain } });
    toast("QR generated");
    loadQrList();
  });

  setInterval(() => {
    loadSummary();
    loadKitchenOrders();
    loadTables();
  }, 3000);
}

async function initWaiterPage() {
  if (!(await ensureRole(["waiter", "admin"]))) return;

  const loadAssigned = async () => {
    try {
      const data = await api("/staff/assigned_tables.php");
      const host = qs("#assignedTables");
      if (!host) return;
      host.innerHTML = data.tables.map(t => `<button class='btn-outline' data-table='${esc(t.table_number)}'>Table ${esc(t.table_number)} (${esc(t.status)})</button>`).join("") || "<p class='muted'>No tables assigned.</p>";
      qsa("[data-table]", host).forEach(btn => btn.addEventListener("click", () => loadOrderForTable(btn.dataset.table)));
    } catch (e) {
      toast(e.message, true);
    }
  };

  const loadOrderForTable = async (tableNumber) => {
    try {
      const data = await api(`/orders/by_table.php?table_number=${encodeURIComponent(tableNumber)}`);
      const host = qs("#waiterOrders");
      if (!host) return;
      if (!data.order) {
        host.innerHTML = `<p class='muted'>No active order for table ${esc(tableNumber)}.</p>`;
        return;
      }
      host.innerHTML = `<div class='card'>
        <h3>Table ${esc(tableNumber)} • Order #${data.order.order_id}</h3>
        ${data.order.items.map(i => `<div class='row' style='justify-content:space-between;padding:6px 0;border-bottom:1px solid #dbe3ef;'>
          <div>${esc(i.name)} x ${i.quantity}</div>
          <select data-w-item='${i.order_item_id}' data-order='${data.order.order_id}'>
            <option value='received' ${i.status==='received'?'selected':''}>Received</option>
            <option value='preparing' ${i.status==='preparing'?'selected':''}>Preparing</option>
            <option value='ready' ${i.status==='ready'?'selected':''}>Ready</option>
            <option value='served' ${i.status==='served'?'selected':''}>Served</option>
          </select>
        </div>`).join("")}
      </div>`;

      qsa("[data-w-item]").forEach(el => el.addEventListener("change", async () => {
        await api("/orders/update_status.php", { method: "POST", body: { order_id: Number(el.dataset.order), order_item_id: Number(el.dataset.wItem), status: el.value } });
        toast("Order item updated");
      }));
    } catch (e) {
      toast(e.message, true);
    }
  };

  qs("#waiterManualChargeBtn")?.addEventListener("click", async () => {
    const order_id = Number(qs("#waiterOrderId").value);
    const title = qs("#waiterChargeTitle").value.trim();
    const amount = Number(qs("#waiterChargeAmount").value);
    await api("/billing/manage.php", { method: "POST", body: { action: "manual_charge", order_id, title, amount } });
    toast("Charge added");
  });

  qs("#waiterCleanBtn")?.addEventListener("click", async () => {
    const table_id = Number(qs("#waiterTableId").value);
    await api("/tables/manage.php", { method: "POST", body: { action: "mark_cleaning", table_id } });
    toast("Table marked for cleaning");
  });

  await loadAssigned();
  await loadTables("#waiterTablesGrid");
  await loadKitchenOrders();

  setInterval(() => {
    loadAssigned();
    loadKitchenOrders();
  }, 3000);
}

function init() {
  attachCommonActions();

  const page = document.body.dataset.page;
  if (page === "index") initIndexPage();
  if (page === "menu") initMenuPage();
  if (page === "kitchen") initKitchenPage();
  if (page === "admin") initAdminPage();
  if (page === "waiter") initWaiterPage();
}

document.addEventListener("DOMContentLoaded", init);