<?php
declare(strict_types=1);

require_once __DIR__ . '/../common/bootstrap.php';
require_method('POST');

$user = require_auth(['admin', 'waiter']);
$input = get_json_input();
$action = clean_string($input['action'] ?? '', 30);

if ($action === 'assign_waiter') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }
    $tableId = (int) ($input['table_id'] ?? 0);
    $waiterId = (int) ($input['waiter_id'] ?? 0);

    $stmt = $pdo->prepare("UPDATE tables SET assigned_waiter_id = :waiter_id WHERE id = :table_id AND restaurant_id = :restaurant_id");
    $stmt->execute([':waiter_id' => $waiterId, ':table_id' => $tableId, ':restaurant_id' => 1]);
    json_success(['message' => 'Waiter assigned']);
}

if ($action === 'merge_tables') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }
    $sourceTableId = (int) ($input['source_table_id'] ?? 0);
    $targetTableId = (int) ($input['target_table_id'] ?? 0);

    if ($sourceTableId <= 0 || $targetTableId <= 0 || $sourceTableId === $targetTableId) {
        json_error('Invalid merge tables payload', 422);
    }

    $stmt = $pdo->prepare("UPDATE tables SET merged_into_table_id = :target_table_id, status = 'occupied' WHERE id = :source_table_id AND restaurant_id = :restaurant_id");
    $stmt->execute([':target_table_id' => $targetTableId, ':source_table_id' => $sourceTableId, ':restaurant_id' => 1]);

    json_success(['message' => 'Tables merged']);
}

if ($action === 'move_order') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }

    $orderId = (int) ($input['order_id'] ?? 0);
    $toTableId = (int) ($input['to_table_id'] ?? 0);

    if ($orderId <= 0 || $toTableId <= 0) {
        json_error('Invalid move payload', 422);
    }

    $pdo->beginTransaction();
    try {
        $currentStmt = $pdo->prepare('SELECT table_id FROM orders WHERE id = :order_id LIMIT 1');
        $currentStmt->execute([':order_id' => $orderId]);
        $current = $currentStmt->fetch();
        if (!$current) {
            throw new RuntimeException('Order not found');
        }

        $fromTableId = (int) $current['table_id'];

        $upd = $pdo->prepare('UPDATE orders SET table_id = :table_id WHERE id = :order_id');
        $upd->execute([':table_id' => $toTableId, ':order_id' => $orderId]);

        $markTo = $pdo->prepare("UPDATE tables SET status = 'occupied' WHERE id = :table_id");
        $markTo->execute([':table_id' => $toTableId]);

        $checkLeft = $pdo->prepare("SELECT COUNT(*) AS total FROM orders WHERE table_id = :table_id AND status IN ('received','preparing','ready','served')");
        $checkLeft->execute([':table_id' => $fromTableId]);
        if ((int) $checkLeft->fetch()['total'] === 0) {
            $markFrom = $pdo->prepare("UPDATE tables SET status = 'available' WHERE id = :table_id");
            $markFrom->execute([':table_id' => $fromTableId]);
        }

        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        json_error($e->getMessage(), 422);
    }

    json_success(['message' => 'Order moved']);
}

if ($action === 'mark_cleaning') {
    $tableId = (int) ($input['table_id'] ?? 0);
    if ($tableId <= 0) {
        json_error('Invalid table ID', 422);
    }

    $stmt = $pdo->prepare("UPDATE tables SET status = 'needs_cleaning' WHERE id = :table_id AND restaurant_id = :restaurant_id");
    $stmt->execute([':table_id' => $tableId, ':restaurant_id' => 1]);
    json_success(['message' => 'Table marked for cleaning']);
}

if ($action === 'close_table') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }

    $tableId = (int) ($input['table_id'] ?? 0);
    if ($tableId <= 0) {
        json_error('Invalid table ID', 422);
    }

    $stmt = $pdo->prepare("UPDATE tables SET status = 'available', merged_into_table_id = NULL WHERE id = :table_id AND restaurant_id = :restaurant_id");
    $stmt->execute([':table_id' => $tableId, ':restaurant_id' => 1]);

    json_success(['message' => 'Table closed and available']);
}

json_error('Unknown table action', 422);