<?php
declare(strict_types=1);

require_once __DIR__ . '/../common/bootstrap.php';
require_once __DIR__ . '/../common/file_guard.php';

$method = $_SERVER['REQUEST_METHOD'];

if ($method === 'GET') {
    require_auth(['admin']);

    $stmt = $pdo->prepare('SELECT id, category_id, name, description, image_path, price, tags, dietary_tags, availability_state, display_order, is_active
                           FROM menu_items WHERE restaurant_id = :restaurant_id ORDER BY category_id ASC, display_order ASC');
    $stmt->execute([':restaurant_id' => 1]);
    $rows = $stmt->fetchAll();

    foreach ($rows as &$row) {
        $row['id'] = (int) $row['id'];
        $row['category_id'] = (int) $row['category_id'];
        $row['price'] = (float) $row['price'];
        $row['display_order'] = (int) $row['display_order'];
        $row['is_active'] = (bool) $row['is_active'];
        $row['name'] = esc_html((string) $row['name']);
        $row['description'] = esc_html((string) ($row['description'] ?? ''));
        $row['tags'] = esc_html((string) ($row['tags'] ?? ''));
        $row['dietary_tags'] = esc_html((string) ($row['dietary_tags'] ?? ''));
        $row['image_path'] = esc_html((string) ($row['image_path'] ?? ''));
    }

    json_success(['items' => $rows]);
}

if ($method === 'POST') {
    require_auth(['admin']);

    $isMultipart = isset($_POST['action']);
    if ($isMultipart) {
        $payload = $_POST;
    } else {
        $payload = get_json_input();
    }

    $action = clean_string($payload['action'] ?? 'create', 20);

    if ($action === 'create') {
        $categoryId = (int) ($payload['category_id'] ?? 0);
        $name = clean_string($payload['name'] ?? '', 150);
        $description = clean_string($payload['description'] ?? '', 500);
        $price = (float) ($payload['price'] ?? 0);
        $tags = clean_string($payload['tags'] ?? '', 255);
        $dietaryTags = clean_string($payload['dietary_tags'] ?? '', 255);
        $availability = clean_string($payload['availability_state'] ?? 'available', 30);
        $displayOrder = (int) ($payload['display_order'] ?? 0);
        $imagePath = null;

        if (isset($_FILES['image'])) {
            $file = ensure_image_upload($_FILES['image']);
            $imagePath = $file['path'];
        }

        if ($categoryId <= 0 || $name === '' || $price <= 0 || !in_array($availability, ['available','out_of_stock','lunch_only','dinner_only','seasonal'], true)) {
            json_error('Invalid item payload', 422);
        }

        $ins = $pdo->prepare('INSERT INTO menu_items (restaurant_id, category_id, name, description, image_path, price, tags, dietary_tags, availability_state, display_order, is_active)
                              VALUES (:restaurant_id, :category_id, :name, :description, :image_path, :price, :tags, :dietary_tags, :availability_state, :display_order, 1)');
        $ins->execute([
            ':restaurant_id' => 1,
            ':category_id' => $categoryId,
            ':name' => $name,
            ':description' => $description,
            ':image_path' => $imagePath,
            ':price' => $price,
            ':tags' => $tags,
            ':dietary_tags' => $dietaryTags,
            ':availability_state' => $availability,
            ':display_order' => $displayOrder
        ]);

        json_success(['menu_item_id' => (int) $pdo->lastInsertId()]);
    }

    if ($action === 'update') {
        $id = (int) ($payload['id'] ?? 0);
        $categoryId = (int) ($payload['category_id'] ?? 0);
        $name = clean_string($payload['name'] ?? '', 150);
        $description = clean_string($payload['description'] ?? '', 500);
        $price = (float) ($payload['price'] ?? 0);
        $tags = clean_string($payload['tags'] ?? '', 255);
        $dietaryTags = clean_string($payload['dietary_tags'] ?? '', 255);
        $availability = clean_string($payload['availability_state'] ?? 'available', 30);
        $displayOrder = (int) ($payload['display_order'] ?? 0);
        $isActive = (int) (!empty($payload['is_active']));

        if ($id <= 0 || $categoryId <= 0 || $name === '' || $price <= 0) {
            json_error('Invalid item payload', 422);
        }

        $imageSet = '';
        $params = [
            ':id' => $id,
            ':category_id' => $categoryId,
            ':name' => $name,
            ':description' => $description,
            ':price' => $price,
            ':tags' => $tags,
            ':dietary_tags' => $dietaryTags,
            ':availability_state' => $availability,
            ':display_order' => $displayOrder,
            ':is_active' => $isActive,
            ':restaurant_id' => 1
        ];

        if (isset($_FILES['image'])) {
            $file = ensure_image_upload($_FILES['image']);
            $imageSet = ', image_path = :image_path';
            $params[':image_path'] = $file['path'];
        }

        $upd = $pdo->prepare('UPDATE menu_items SET category_id = :category_id, name = :name, description = :description, price = :price, tags = :tags, dietary_tags = :dietary_tags,
                              availability_state = :availability_state, display_order = :display_order, is_active = :is_active' . $imageSet . '
                              WHERE id = :id AND restaurant_id = :restaurant_id');
        $upd->execute($params);

        json_success(['message' => 'Item updated']);
    }

    if ($action === 'delete') {
        $id = (int) ($payload['id'] ?? 0);
        if ($id <= 0) {
            json_error('Invalid item ID', 422);
        }

        $del = $pdo->prepare('DELETE FROM menu_items WHERE id = :id AND restaurant_id = :restaurant_id');
        $del->execute([':id' => $id, ':restaurant_id' => 1]);

        json_success(['message' => 'Item deleted']);
    }

    if ($action === 'set_schedule') {
        $menuItemId = (int) ($payload['menu_item_id'] ?? 0);
        $schedules = $payload['schedules'] ?? [];

        if ($menuItemId <= 0 || !is_array($schedules)) {
            json_error('Invalid schedule payload', 422);
        }

        $pdo->beginTransaction();
        try {
            $del = $pdo->prepare('DELETE FROM item_availability_schedules WHERE menu_item_id = :menu_item_id');
            $del->execute([':menu_item_id' => $menuItemId]);

            $ins = $pdo->prepare('INSERT INTO item_availability_schedules (menu_item_id, day_of_week, start_time, end_time) VALUES (:menu_item_id, :day_of_week, :start_time, :end_time)');
            foreach ($schedules as $schedule) {
                $day = (int) ($schedule['day_of_week'] ?? -1);
                $start = clean_string($schedule['start_time'] ?? '', 8);
                $end = clean_string($schedule['end_time'] ?? '', 8);

                if ($day < 0 || $day > 6 || $start === '' || $end === '') {
                    throw new RuntimeException('Invalid schedule entry');
                }

                $ins->execute([
                    ':menu_item_id' => $menuItemId,
                    ':day_of_week' => $day,
                    ':start_time' => $start,
                    ':end_time' => $end
                ]);
            }

            $pdo->commit();
        } catch (Throwable $e) {
            $pdo->rollBack();
            json_error($e->getMessage(), 422);
        }

        json_success(['message' => 'Schedule updated']);
    }
}

json_error('Method not allowed', 405);