<?php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if (session_status() === PHP_SESSION_NONE) {
    session_set_cookie_params([
        'httponly' => true,
        'samesite' => 'Lax',
        'secure' => (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off')
    ]);
    session_start();
}

require_once __DIR__ . '/../../config.php';

function json_success(array $data = []): void
{
    echo json_encode(['status' => 'success', 'data' => $data]);
    exit;
}

function json_error(string $message, int $statusCode = 400): void
{
    http_response_code($statusCode);
    echo json_encode(['status' => 'error', 'message' => $message]);
    exit;
}

function get_json_input(): array
{
    $raw = file_get_contents('php://input');
    if ($raw === false || $raw === '') {
        return [];
    }

    $data = json_decode($raw, true);
    if (!is_array($data)) {
        json_error('Invalid JSON payload', 422);
    }

    return $data;
}

function clean_string(?string $value, int $maxLen = 255): string
{
    $value = trim((string) $value);
    $value = strip_tags($value);
    if (mb_strlen($value) > $maxLen) {
        $value = mb_substr($value, 0, $maxLen);
    }
    return $value;
}

function require_method(string $method): void
{
    if ($_SERVER['REQUEST_METHOD'] !== strtoupper($method)) {
        json_error('Method not allowed', 405);
    }
}

function current_user(): ?array
{
    return $_SESSION['user'] ?? null;
}

function require_auth(array $allowedRoles = []): array
{
    $user = current_user();
    if (!$user) {
        json_error('Authentication required', 401);
    }

    if ($allowedRoles && !in_array($user['role'], $allowedRoles, true)) {
        json_error('Forbidden', 403);
    }

    return $user;
}

function esc_html(string $value): string
{
    return htmlspecialchars($value, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function get_setting(PDO $pdo, int $restaurantId, string $key, string $default = '0'): string
{
    $stmt = $pdo->prepare('SELECT setting_value FROM settings WHERE restaurant_id = :restaurant_id AND setting_key = :setting_key LIMIT 1');
    $stmt->execute([
        ':restaurant_id' => $restaurantId,
        ':setting_key' => $key
    ]);
    $row = $stmt->fetch();
    return $row['setting_value'] ?? $default;
}

function recalculate_bill(PDO $pdo, int $orderId): array
{
    $orderStmt = $pdo->prepare('SELECT id FROM orders WHERE id = :order_id LIMIT 1');
    $orderStmt->execute([':order_id' => $orderId]);
    $order = $orderStmt->fetch();
    if (!$order) {
        json_error('Order not found', 404);
    }

    $subtotalStmt = $pdo->prepare('SELECT COALESCE(SUM(line_total), 0) AS subtotal FROM order_items WHERE order_id = :order_id');
    $subtotalStmt->execute([':order_id' => $orderId]);
    $subtotal = (float) $subtotalStmt->fetch()['subtotal'];

    $manualStmt = $pdo->prepare('SELECT COALESCE(SUM(amount), 0) AS manual_total FROM manual_charges WHERE order_id = :order_id');
    $manualStmt->execute([':order_id' => $orderId]);
    $manualTotal = (float) $manualStmt->fetch()['manual_total'];

    $restStmt = $pdo->prepare('SELECT restaurant_id FROM orders WHERE id = :order_id LIMIT 1');
    $restStmt->execute([':order_id' => $orderId]);
    $restaurantId = (int) $restStmt->fetch()['restaurant_id'];

    $taxPercent = (float) get_setting($pdo, $restaurantId, 'tax_percent', '0');
    $servicePercent = (float) get_setting($pdo, $restaurantId, 'service_charge_percent', '0');

    $discountStmt = $pdo->prepare('SELECT discount_amount FROM bills WHERE order_id = :order_id LIMIT 1');
    $discountStmt->execute([':order_id' => $orderId]);
    $discountAmount = (float) (($discountStmt->fetch()['discount_amount'] ?? 0));

    $taxable = max(0, ($subtotal + $manualTotal - $discountAmount));
    $taxAmount = round($taxable * ($taxPercent / 100), 2);
    $serviceAmount = round($taxable * ($servicePercent / 100), 2);
    $grandTotal = round($taxable + $taxAmount + $serviceAmount, 2);

    $upsertSql = 'INSERT INTO bills (order_id, subtotal, manual_charges_total, discount_amount, tax_percent, tax_amount, service_charge_percent, service_charge_amount, grand_total)
                  VALUES (:order_id, :subtotal, :manual_total, :discount_amount, :tax_percent, :tax_amount, :service_percent, :service_amount, :grand_total)
                  ON DUPLICATE KEY UPDATE
                    subtotal = VALUES(subtotal),
                    manual_charges_total = VALUES(manual_charges_total),
                    discount_amount = VALUES(discount_amount),
                    tax_percent = VALUES(tax_percent),
                    tax_amount = VALUES(tax_amount),
                    service_charge_percent = VALUES(service_charge_percent),
                    service_charge_amount = VALUES(service_charge_amount),
                    grand_total = VALUES(grand_total),
                    updated_at = CURRENT_TIMESTAMP';

    $upsertStmt = $pdo->prepare($upsertSql);
    $upsertStmt->execute([
        ':order_id' => $orderId,
        ':subtotal' => $subtotal,
        ':manual_total' => $manualTotal,
        ':discount_amount' => $discountAmount,
        ':tax_percent' => $taxPercent,
        ':tax_amount' => $taxAmount,
        ':service_percent' => $servicePercent,
        ':service_amount' => $serviceAmount,
        ':grand_total' => $grandTotal
    ]);

    return [
        'subtotal' => $subtotal,
        'manual_charges_total' => $manualTotal,
        'discount_amount' => $discountAmount,
        'tax_percent' => $taxPercent,
        'tax_amount' => $taxAmount,
        'service_charge_percent' => $servicePercent,
        'service_charge_amount' => $serviceAmount,
        'grand_total' => $grandTotal
    ];
}