<?php
declare(strict_types=1);

require_once __DIR__ . '/../common/bootstrap.php';
require_method('POST');

$user = require_auth(['admin', 'waiter']);
$input = get_json_input();
$action = clean_string($input['action'] ?? '', 30);
$orderId = (int) ($input['order_id'] ?? 0);

if ($orderId <= 0) {
    json_error('order_id is required', 422);
}

if ($action === 'manual_charge') {
    $title = clean_string($input['title'] ?? '', 120);
    $amount = (float) ($input['amount'] ?? 0);

    if ($title === '' || $amount <= 0) {
        json_error('Invalid manual charge', 422);
    }

    $ins = $pdo->prepare('INSERT INTO manual_charges (order_id, added_by_staff_id, title, amount) VALUES (:order_id, :staff_user_id, :title, :amount)');
    $ins->execute([
        ':order_id' => $orderId,
        ':staff_user_id' => $user['id'],
        ':title' => $title,
        ':amount' => $amount
    ]);

    $totals = recalculate_bill($pdo, $orderId);
    json_success(['message' => 'Manual charge added', 'totals' => $totals]);
}

if ($action === 'set_discount') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }

    $discount = (float) ($input['discount_amount'] ?? 0);
    if ($discount < 0) {
        json_error('Invalid discount amount', 422);
    }

    $upsert = $pdo->prepare('INSERT INTO bills (order_id, discount_amount) VALUES (:order_id, :discount_amount)
                             ON DUPLICATE KEY UPDATE discount_amount = VALUES(discount_amount), updated_at = CURRENT_TIMESTAMP');
    $upsert->execute([
        ':order_id' => $orderId,
        ':discount_amount' => $discount
    ]);

    $totals = recalculate_bill($pdo, $orderId);
    json_success(['message' => 'Discount updated', 'totals' => $totals]);
}

if ($action === 'set_settings') {
    if ($user['role'] !== 'admin') {
        json_error('Forbidden', 403);
    }

    $taxPercent = (float) ($input['tax_percent'] ?? 0);
    $servicePercent = (float) ($input['service_charge_percent'] ?? 0);

    if ($taxPercent < 0 || $taxPercent > 100 || $servicePercent < 0 || $servicePercent > 100) {
        json_error('Invalid tax or service charge percent', 422);
    }

    $upsert = $pdo->prepare('INSERT INTO settings (restaurant_id, setting_key, setting_value)
                             VALUES (:restaurant_id, :setting_key, :setting_value)
                             ON DUPLICATE KEY UPDATE setting_value = VALUES(setting_value), updated_at = CURRENT_TIMESTAMP');

    $upsert->execute([
        ':restaurant_id' => 1,
        ':setting_key' => 'tax_percent',
        ':setting_value' => (string) $taxPercent
    ]);

    $upsert->execute([
        ':restaurant_id' => 1,
        ':setting_key' => 'service_charge_percent',
        ':setting_value' => (string) $servicePercent
    ]);

    $totals = recalculate_bill($pdo, $orderId);
    json_success(['message' => 'Billing settings updated', 'totals' => $totals]);
}

if ($action === 'set_split') {
    $splitCount = (int) ($input['split_count'] ?? 1);
    if ($splitCount <= 0) {
        json_error('Invalid split count', 422);
    }

    $upsert = $pdo->prepare('INSERT INTO bills (order_id, split_count) VALUES (:order_id, :split_count)
                             ON DUPLICATE KEY UPDATE split_count = VALUES(split_count), updated_at = CURRENT_TIMESTAMP');
    $upsert->execute([
        ':order_id' => $orderId,
        ':split_count' => $splitCount
    ]);

    json_success(['message' => 'Split updated']);
}

if ($action === 'mark_paid') {
    $method = clean_string($input['payment_method'] ?? '', 20);
    $amount = (float) ($input['amount'] ?? 0);
    $referenceNote = clean_string($input['reference_note'] ?? '', 255);

    if (!in_array($method, ['cash', 'card', 'upi', 'other'], true) || $amount <= 0) {
        json_error('Invalid payment payload', 422);
    }

    $totals = recalculate_bill($pdo, $orderId);
    $grandTotal = (float) $totals['grand_total'];

    $billStmt = $pdo->prepare('SELECT id FROM bills WHERE order_id = :order_id LIMIT 1');
    $billStmt->execute([':order_id' => $orderId]);
    $bill = $billStmt->fetch();
    if (!$bill) {
        json_error('Bill not found', 404);
    }

    $billId = (int) $bill['id'];

    $pdo->beginTransaction();
    try {
        $payIns = $pdo->prepare('INSERT INTO payments (bill_id, paid_by_staff_id, amount, method, reference_note, paid_at)
                                 VALUES (:bill_id, :staff_user_id, :amount, :method, :reference_note, NOW())');
        $payIns->execute([
            ':bill_id' => $billId,
            ':staff_user_id' => $user['id'],
            ':amount' => $amount,
            ':method' => $method,
            ':reference_note' => $referenceNote
        ]);

        $sumStmt = $pdo->prepare('SELECT COALESCE(SUM(amount), 0) AS paid_total FROM payments WHERE bill_id = :bill_id');
        $sumStmt->execute([':bill_id' => $billId]);
        $paidTotal = (float) $sumStmt->fetch()['paid_total'];

        if ($paidTotal + 0.001 >= $grandTotal) {
            $markBill = $pdo->prepare('UPDATE bills SET is_paid = 1, closed_at = NOW() WHERE id = :bill_id');
            $markBill->execute([':bill_id' => $billId]);

            $markOrder = $pdo->prepare("UPDATE orders SET status = 'closed' WHERE id = :order_id");
            $markOrder->execute([':order_id' => $orderId]);

            $tableStmt = $pdo->prepare('SELECT table_id FROM orders WHERE id = :order_id LIMIT 1');
            $tableStmt->execute([':order_id' => $orderId]);
            $tableId = (int) $tableStmt->fetch()['table_id'];

            $markTable = $pdo->prepare("UPDATE tables SET status = 'needs_cleaning' WHERE id = :table_id");
            $markTable->execute([':table_id' => $tableId]);
        }

        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        json_error($e->getMessage(), 422);
    }

    json_success(['message' => 'Payment recorded', 'grand_total' => $grandTotal]);
}

json_error('Unknown billing action', 422);