<?php
declare(strict_types=1);

require_once __DIR__ . '/../common/bootstrap.php';
require_method('GET');
require_auth(['admin']);

$from = clean_string($_GET['from'] ?? date('Y-m-01'), 20);
$to = clean_string($_GET['to'] ?? date('Y-m-d'), 20);

$sql = "SELECT o.id AS order_id, t.table_number, COALESCE(b.grand_total,0) AS grand_total, o.created_at
        FROM orders o
        INNER JOIN tables t ON t.id = o.table_id
        LEFT JOIN bills b ON b.order_id = o.id
        WHERE DATE(o.created_at) BETWEEN :from_date AND :to_date
        ORDER BY o.created_at DESC
        LIMIT 200";
$stmt = $pdo->prepare($sql);
$stmt->execute([':from_date' => $from, ':to_date' => $to]);
$rows = $stmt->fetchAll();

$contentLines = [
    'Cafe QR - Orders Report',
    'Date Range: ' . $from . ' to ' . $to,
    '----------------------------------------'
];

$total = 0.0;
foreach ($rows as $row) {
    $line = 'Order #' . $row['order_id'] . ' | Table ' . $row['table_number'] . ' | Rs ' . number_format((float) $row['grand_total'], 2);
    $contentLines[] = $line;
    $total += (float) $row['grand_total'];
}
$contentLines[] = '----------------------------------------';
$contentLines[] = 'Total Revenue: Rs ' . number_format($total, 2);

$objects = [];
$offsets = [];

$pdfHeader = "%PDF-1.4\n";
$objects[] = "1 0 obj << /Type /Catalog /Pages 2 0 R >> endobj\n";
$objects[] = "2 0 obj << /Type /Pages /Kids [3 0 R] /Count 1 >> endobj\n";
$objects[] = "3 0 obj << /Type /Page /Parent 2 0 R /MediaBox [0 0 595 842] /Contents 4 0 R /Resources << /Font << /F1 5 0 R >> >> >> endobj\n";

$y = 800;
$stream = "BT\n/F1 12 Tf\n";
foreach ($contentLines as $line) {
    $safe = str_replace(['\\', '(', ')'], ['\\\\', '\\(', '\\)'], $line);
    $stream .= "50 {$y} Td ({$safe}) Tj\n";
    $stream .= "-50 0 Td\n";
    $y -= 18;
    if ($y < 50) {
        break;
    }
}
$stream .= "ET";
$objects[] = '4 0 obj << /Length ' . strlen($stream) . " >> stream\n" . $stream . "\nendstream endobj\n";
$objects[] = "5 0 obj << /Type /Font /Subtype /Type1 /BaseFont /Helvetica >> endobj\n";

$body = '';
foreach ($objects as $obj) {
    $offsets[] = strlen($pdfHeader . $body);
    $body .= $obj;
}

$xrefPos = strlen($pdfHeader . $body);
$xref = "xref\n0 " . (count($objects) + 1) . "\n";
$xref .= "0000000000 65535 f \n";
foreach ($offsets as $offset) {
    $xref .= str_pad((string) $offset, 10, '0', STR_PAD_LEFT) . " 00000 n \n";
}

$trailer = "trailer << /Size " . (count($objects) + 1) . " /Root 1 0 R >>\nstartxref\n{$xrefPos}\n%%EOF";

$pdf = $pdfHeader . $body . $xref . $trailer;

json_success([
    'file_name' => 'orders_report.pdf',
    'mime_type' => 'application/pdf',
    'base64_content' => base64_encode($pdf)
]);